#include <p30fxxxx.h>
#include "spi.h"

/********************************************************************
*     Function Name :  CloseSPI1                                    *
*     Description   :  This routine disables the SPI module and its *
*                      interrupt bits.                              *
*     Parameters    :  None                                         *
*     Return Value  :  None                                         *
********************************************************************/
void CloseSPI1()
{ 
  
      IEC0bits.SPI1IE = 0;  /* Disable the Interrupt bit in the 
                               Interrupt Enable Control Register */
      SPI1STATbits.SPIEN = 0;
                            /*  Disable the module. All pins controlled
                                by PORT Functions */
      IFS0bits.SPI1IF = 0;  /* Disable the Interrupt flag bit in the
                               Interrupt Flag Control Register */    
}

/************************************************************************
*     Function Name :  ConfigIntSPI1                                    *
*     Description   :  This Function Configures Interrupt and sets      *
*                      Interrupt Priority                               *
*     Parameters    :  unsigned int config                              *
*     Return Value  :  None                                             *
*************************************************************************/
void ConfigIntSPI1( unsigned int config)
{
    IFS0bits.SPI1IF = 0;                   /* Clear IF bit  */
    IPC2bits.SPI1IP = (config &0x0007);    /* assign interrupt priority to 
                                             IPC register */

    IEC0bits.SPI1IE = (config &0x0008)>>3; /*Interrupt Enable/Disable bit 
                                             in IEC Register */
}

/********************************************************************
*     Function Name : DataRdySPI1                                   *
*     Description   : Determine if there is a byte/word to be read  *
*                     from the SPIBUF register.                     *
*     Parameters    : None                                          *
*     Return Value  : status bit to indicate if RBF = 1 else 0      *
********************************************************************/
char DataRdySPI1()
{   
    return SPI1STATbits.SPIRBF; /* return the RBF bit status */
}

/***************************************************************************
*     Function Name :  getsSPI1                                            *
*     Description   :  This routine reads a string from the SPI            *
*                      receive buffer.The number of byte/word to be read   *
                       is determined by parameter 'length'                 *
*     Parameters    :  unsigned int length: The length of data expected    *
*                      unsigned int *rdptr   the received data to be 
*                      recorded to this array
*                      unsigned int uart_data_wait timeout value
*     Return Value  :  unsigned int number of data bytes yet to be received*                     
****************************************************************************/
unsigned int getsSPI1(unsigned int length, unsigned int *rdptr, unsigned int spi_data_wait)
{
    int wait = 0;
    char *temp_ptr = (char *)rdptr;
    while (length)                     /* stay in loop until length = 0  */
    {
        while(!DataRdySPI1())
        {

           if(wait < spi_data_wait)
               wait++ ;                /* wait for time out operation */
           else
               return(length);         /*Time out, return number of byte/word to be read */
        }
        wait = 0;
        
        if(SPI1CONbits.MODE16)
            *rdptr++ = getcSPI1();     /* read a single word */
        else
            *temp_ptr++ = getcSPI1();  /* read a single byte */
        length--;                      /* reduce string length count by 1*/
    }
    return (length);                   /* return number of byte/word to be read i.e, 0 */
}

/********************************************************************************
*   Function Name :  OpenSPI1                                                   *
*   Description   :  This functionin Initializes the SPI module and Also enables*
                     SPI module 
*   Parameters    :  unsigned int config1,unsigned int config2                  *
*   Return Value  :  None                                                       *
*   Note:            Config1 contains information about framed SPI support,     *
                     frame sync pulse,Disable SDO pin, Word/byte communication  *  
                     data input sample phase bit, clock edge select bit,slave   *
*                    select ,clock polarity select bit ,Master enable bit       *
*                    secondary prescale bit, Primary prescale bit               *
*                    Config2 contains information about Enable/Disable SPI,SPI  *
                     Operation during CPU IDLE mode                             *
*********************************************************************************/
void OpenSPI1(unsigned int config1,unsigned int config2)
{       
     SPI1CON = config1;      /* Initalizes the spi module */
     SPI1STAT = config2;     /* Enable/Disable the spi module */
}

/*****************************************************************************
*     Function Name :  putsSPI1                                              *
*     Description   :  This function writes the specified length of data     *
                       word/byte to be transmitted into the transmit buffer  *
*     Parameters    :  unsigned int: length of bytes/words to be written     *
                       unsigned int*:  Address ofthe location where the data *
                       to be transmitted is stored                           *
*     Return Value  :  None                                                  *                                                                
******************************************************************************/
void putsSPI1(unsigned int length, unsigned int *wrptr)
{  
    char *temp_ptr = (char *) wrptr;
    while (length)                   /* write byte/word until length is 0 */
    {            
        if(SPI1CONbits.MODE16)
            SPI1BUF = *wrptr++;      /* initiate SPI bus cycle by word write */ 
        else
            SPI1BUF = *temp_ptr++;   /* initiate SPI bus cycle by byte write */ 
        while(SPI1STATbits.SPITBF);  /* wait until 'SPITBF' bit is cleared */
        length--;                    /* decrement length */
    }
}

/******************************************************************************
*     Function Name :   ReadSPI1                                              *
*     Description   :   This function will read single byte/ word  from SPI   *
                        bus. If SPI is configured for byte  communication     *
                        then upper byte of SPIBUF is masked.                  *         
*     Parameters    :   None                                                  *
*     Return Value  :   contents of SPIBUF register                           *
*******************************************************************************/
unsigned int ReadSPI1()
{         
    /* Check for Receive buffer full status bit of status register*/
    if (SPI1STATbits.SPIRBF)
    { 
        SPI1STATbits.SPIROV = 0;
                
        if (SPI1CONbits.MODE16)
            return (SPI1BUF);           /* return word read */
        else
            return (SPI1BUF & 0xff);    /* return byte read */
    }
    return -1;                  		/* RBF bit is not set return error*/
}

/********************************************************************
*     Function Name : WriteSPI1                                     *
*     Description   : This routine writes a single byte/word to     * 
*                     the SPI bus.                                  *
*     Parameters    : Single data byte/word for SPI bus             *
*     Return Value  : None                                          *
********************************************************************/
void WriteSPI1(unsigned int data_out)
{   
    if (SPI1CONbits.MODE16)          /* word write */
        SPI1BUF = data_out;
    else 
        SPI1BUF = data_out & 0xff;   /*  byte write  */
}

/* The following devices support SPI2 */
#if defined(__dsPIC30F6010__) || defined(__dsPIC30F5011__)  || defined(__dsPIC30F6011__) || \
    defined(__dsPIC30F6012__) || defined(__dsPIC30F5013__) ||  defined(__dsPIC30F6013__) || \
    defined(__dsPIC30F6014__) || defined(__dsPIC30F5015__) 

/********************************************************************
*     Function Name :  CloseSPI2                                    *
*     Description   :  This routine disables the SPI module and its *
                       interrupt bits.   
*     Parameters    :  None                                         *
*     Return Value  :  None                                         *
********************************************************************/
void CloseSPI2( )
{
      IEC1bits.SPI2IE = 0;  /*  Disable the Interrupt bit in the 
                                Interrupt Enable Control Register */
      SPI2STATbits.SPIEN = 0;
                            /*  Disable the module. All pins controlled
                                by PORT Functions */
      IFS1bits.SPI2IF = 0;  /*  Disable the Interrupt flag bit in the 
                                Interrupt Flag Control Register */    
}

/************************************************************************
*     Function Name :  ConfigIntSPI2                                    *
*     Description   :  This Function Configures Interrupt and sets      *
*                      Interrupt Priority                               *
*     Parameters    :  unsigned int config                              *
*     Return Value  :  None                                             *
************************************************************************/
void ConfigIntSPI2( unsigned int config)
{
    IFS1bits.SPI2IF = 0;               /* Clear IF bit  */
    IPC6bits.SPI2IP=(config &0x0007);  /* assign interrupt priority to 
                                          IPC register */

    IEC1bits.SPI2IE=(config &0x0008)>>3;/*Interrupt Enable/Disable bit 
                                          IEC Register */
        
}

/********************************************************************
*     Function Name : DataRdySPI2                                   *
*     Description   : Determine if there is a byte/word to be read  *
*                     from the SPIBUF register.                     *
*     Parameters    : None                                          *
*     Return Value  : status bit to indicate if RBF = 1 else 0      *
********************************************************************/
char DataRdySPI2()
{   
    return SPI2STATbits.SPIRBF; /* return RBF bit status */
}

/***************************************************************************
*     Function Name :  getsSPI2                                            *
*     Description   :  This routine reads a string from the SPI            *
*                      receive buffer.The number of byte/word to be read   *
                       is determined by parameter 'length'                 *
*     Parameters    :  unsigned int length: The length of data expected    *
*                      unsigned int *rdptr   the received data to be 
*                      recorded to this array
*                      unsigned int uart_data_wait timeout value
*     Return Value  :  unsigned int number of data bytes yet to be received*                     
****************************************************************************/
unsigned int getsSPI2(unsigned int length, unsigned int *rdptr, unsigned int spi_data_wait)
{
    int wait = 0;
    char *temp_ptr = (char *)rdptr;
    while (length)                         /* stay in loop until length = 0  */
    {
        while(!DataRdySPI2())
        {

           if(wait < spi_data_wait)
              wait++ ;                     /* wait for time out operation */
           else
              return(length);              /* Time out, return number of byte/word read 
                                              for a given length */
        }
        wait = 0;
        if(SPI2CONbits.MODE16)
            *rdptr++ = getcSPI2();         /* read a single word */
        else
            *temp_ptr++ = getcSPI2();      /* read a single byte */
        length--;                          /* reduce string length count by 1*/
    }

    return (length);                       /* return number of byte/word to be read i.e, 0 */
}

/********************************************************************************
*   Function Name :  OpenSPI2                                                   *
*   Description   :  This function Initializes the SPI module and Also enables  *
                     SPI module 
*   Parameters    :  unsigned int config1,unsigned int config2                  *
*   Return Value  :  None                                                       *
*   Note:            Config1 contains information about framed SPI support,     *
                     frame sync pulse,Disable SDO pin, Word/byte communication  *  
                     data input sample phase bit, clock edge select bit,slave   *
*                    select ,clock polarity select bit ,Master enable bit       *
*                    secondary prescale bit, Primary prescale bit               *
*                    Config2 contains information about Enable/Disable SPI,SPI  *
                     Operation during CPU IDLE mode                             *
*********************************************************************************/
void OpenSPI2(unsigned int config1,unsigned int config2)
{   
     SPI2CON = config1;   /* Initalizes the spi module */ 
     SPI2STAT = config2;  /* Enable/Disable the spi module */ 
}

/*****************************************************************************
*     Function Name :  putsSPI2                                              *
*     Description   :  This function writes the specified length of data     *
                       word/byte to be transmitted into the transmit buffer  *
*     Parameters    :  unsigned int: length of bytes/words to be written     *
                       unsigned int*:  Address ofthe location where the data *
                       to be transmitted is stored                           *
*     Return Value  :  None                                                  *                                                                
******************************************************************************/
void putsSPI2(unsigned int length, unsigned int *wrptr)
{  
    char *temp_ptr = (char *) wrptr;
    while (length)                   /* write byte/word until length is 0 */
    {           
        if(SPI2CONbits.MODE16)
            SPI2BUF = *wrptr++;      /* initiate SPI bus cycle by word write */ 
        else
            SPI2BUF = *temp_ptr++;   /* initiate SPI bus cycle by byte write */ 
        while(SPI2STATbits.SPITBF);  /* wait until 'SPITBF' bit is cleared */
        length--;                    /* decrement length */
    }
}

/******************************************************************************
*     Function Name :   ReadSPI2                                              *
*     Description   :   This function will read single byte/ word  from SPI   *
                        bus. If SPI is configured for byte  communication     *
                        then upper byte of SPIBUF is masked.                  *         
*     Parameters    :   None                                                  *
*     Return Value  :   contents of SPIBUF register                           *
*******************************************************************************/
unsigned int ReadSPI2()
{         
    /* Check for Receive buffer full status bit of status register*/
    if (SPI2STATbits.SPIRBF)
    {
        SPI2STATbits.SPIROV = 0;
                
        if (SPI2CONbits.MODE16)
            return ( SPI2BUF );        /* return word read */
        else
            return (SPI2BUF & 0xff);   /* return byte read */
    }
	return -1;                         /* RBF bit is not set return error*/
}

/********************************************************************
*     Function Name : WriteSPI2                                     *
*     Description   : This routine writes a single byte/word to     * 
*                     the SPI bus.                                  *
*     Parameters    : Single data byte/word for SPI bus             *
*     Return Value  : None                                          *
********************************************************************/
void WriteSPI2(unsigned int data_out)
{   
    if (SPI2CONbits.MODE16)           /* word write */
        SPI2BUF = data_out;
    else 
        SPI2BUF = data_out & 0xff;	  /*  byte write  */
}

#endif


